# frozen_string_literal: true

require 'capybara/selector/selector'
Capybara::Selector::FilterSet.add(:_field) do
  node_filter(:checked, :boolean) { |node, value| !(value ^ node.checked?) }
  node_filter(:unchecked, :boolean) { |node, value| (value ^ node.checked?) }
  node_filter(:disabled, :boolean, default: false, skip_if: :all) { |node, value| !(value ^ node.disabled?) }
  node_filter(:multiple, :boolean) { |node, value| !(value ^ node.multiple?) }

  expression_filter(:name) { |xpath, val| xpath[XPath.attr(:name) == val] }
  expression_filter(:placeholder) { |xpath, val| xpath[XPath.attr(:placeholder) == val] }

  describe(:node_filters) do |checked: nil, unchecked: nil, disabled: nil, multiple: nil, **|
    desc, states = +'', []
    states << 'checked' if checked || (unchecked == false)
    states << 'not checked' if unchecked || (checked == false)
    states << 'disabled' if disabled == true
    states << 'not disabled' if disabled == false
    desc << " that is #{states.join(' and ')}" unless states.empty?
    desc << ' with the multiple attribute' if multiple == true
    desc << ' without the multiple attribute' if multiple == false
    desc
  end
end

# rubocop:disable Metrics/BlockLength

Capybara.add_selector(:xpath) do
  xpath { |xpath| xpath }
end

Capybara.add_selector(:css) do
  css { |css| css }
end

Capybara.add_selector(:id) do
  xpath { |id| builder(XPath.descendant).add_attribute_conditions(id: id) }
  locator_filter {  |node, id| id.is_a?(Regexp) ? node[:id] =~ id : true }
end

Capybara.add_selector(:field) do
  visible { |options| :hidden if options[:type] == 'hidden' }
  xpath do |locator, **options|
    invalid_types = %w[submit image]
    invalid_types << 'hidden' unless options[:type] == 'hidden'
    xpath = XPath.descendant(:input, :textarea, :select)[!XPath.attr(:type).one_of(*invalid_types)]
    locate_field(xpath, locator, options)
  end

  expression_filter(:type) do |expr, type|
    type = type.to_s
    if %w[textarea select].include?(type)
      expr.self(type.to_sym)
    else
      expr[XPath.attr(:type) == type]
    end
  end

  filter_set(:_field) # checked/unchecked/disabled/multiple/name/placeholder

  node_filter(:readonly, :boolean) { |node, value| !(value ^ node.readonly?) }
  node_filter(:with) do |node, with|
    val = node.value
    (with.is_a?(Regexp) ? val =~ with : val == with.to_s).tap do |res|
      add_error("Expected value to be #{with.inspect} but was #{val.inspect}") unless res
    end
  end

  describe_expression_filters do |type: nil, **options|
    desc = +''
    (expression_filters.keys & options.keys).each { |ef| desc << " with #{ef} #{options[ef]}" }
    desc << " of type #{type.inspect}" if type
    desc
  end

  describe_node_filters do |**options|
    " with value #{options[:with].to_s.inspect}" if options.key?(:with)
  end
end

Capybara.add_selector(:fieldset) do
  xpath do |locator, legend: nil, **|
    locator_matchers = (XPath.attr(:id) == locator.to_s) | XPath.child(:legend)[XPath.string.n.is(locator.to_s)]
    locator_matchers |= XPath.attr(test_id) == locator.to_s if test_id
    xpath = XPath.descendant(:fieldset)[locator && locator_matchers]
    xpath = xpath[XPath.child(:legend)[XPath.string.n.is(legend)]] if legend
    xpath
  end

  node_filter(:disabled, :boolean) { |node, value| !(value ^ node.disabled?) }
end

Capybara.add_selector(:link) do
  xpath do |locator, href: true, alt: nil, title: nil, **|
    xpath = builder(XPath.descendant(:a)).add_attribute_conditions(href: href)

    unless locator.nil?
      locator = locator.to_s
      matchers = [XPath.attr(:id) == locator,
                  XPath.string.n.is(locator),
                  XPath.attr(:title).is(locator),
                  XPath.descendant(:img)[XPath.attr(:alt).is(locator)]]
      matchers << XPath.attr(:'aria-label').is(locator) if enable_aria_label
      matchers << XPath.attr(test_id) == locator if test_id
      xpath = xpath[matchers.reduce(:|)]
    end

    xpath = xpath[find_by_attr(:title, title)]
    xpath = xpath[XPath.descendant(:img)[XPath.attr(:alt) == alt]] if alt
    xpath
  end

  node_filter(:href) do |node, href|
    # If not a Regexp it's been handled in the main XPath
    (href.is_a?(Regexp) ? node[:href].match(href) : true).tap do |res|
      add_error "Expected href to match #{href.inspect} but it was #{node[:href].inspect}" unless res
    end
  end

  expression_filter(:download, valid_values: [true, false, String]) do |expr, download|
    builder(expr).add_attribute_conditions(download: download)
  end

  describe_expression_filters do |**options|
    desc = +''
    if (href = options[:href])
      desc << " with href #{'matching ' if href.is_a? Regexp}#{href.inspect}"
    elsif options.key?(:href) # is nil/false specified?
      desc << ' with no href attribute'
    end
    desc
  end
end

Capybara.add_selector(:button) do
  xpath(:value, :title, :type) do |locator, **options|
    input_btn_xpath = XPath.descendant(:input)[XPath.attr(:type).one_of('submit', 'reset', 'image', 'button')]
    btn_xpath = XPath.descendant(:button)
    image_btn_xpath = XPath.descendant(:input)[XPath.attr(:type) == 'image']

    unless locator.nil?
      locator = locator.to_s
      locator_matchers = XPath.attr(:id).equals(locator) | XPath.attr(:value).is(locator) | XPath.attr(:title).is(locator)
      locator_matchers |= XPath.attr(:'aria-label').is(locator) if enable_aria_label
      locator_matchers |= XPath.attr(test_id) == locator if test_id

      input_btn_xpath = input_btn_xpath[locator_matchers]

      btn_xpath = btn_xpath[locator_matchers | XPath.string.n.is(locator) | XPath.descendant(:img)[XPath.attr(:alt).is(locator)]]

      alt_matches = XPath.attr(:alt).is(locator)
      alt_matches |= XPath.attr(:'aria-label').is(locator) if enable_aria_label
      image_btn_xpath = image_btn_xpath[alt_matches]
    end

    %i[value title type].inject(input_btn_xpath.union(btn_xpath).union(image_btn_xpath)) do |memo, ef|
      memo[find_by_attr(ef, options[ef])]
    end
  end

  node_filter(:disabled, :boolean, default: false, skip_if: :all) { |node, value| !(value ^ node.disabled?) }

  describe_expression_filters
  describe_node_filters do |disabled: nil, **|
    ' that is disabled' if disabled == true
  end
end

Capybara.add_selector(:link_or_button) do
  label 'link or button'
  xpath do |locator, **options|
    self.class.all.values_at(:link, :button).map do |selector|
      instance_exec(locator, options, &selector.xpath)
    end.reduce(:union)
  end

  node_filter(:disabled, :boolean, default: false, skip_if: :all) { |node, value| node.tag_name == 'a' || !(value ^ node.disabled?) }

  describe_node_filters do |disabled: nil, **|
    ' that is disabled' if disabled == true
  end
end

Capybara.add_selector(:fillable_field) do
  label 'field'

  xpath do |locator, allow_self: nil, **options|
    xpath = XPath.axis(allow_self ? :"descendant-or-self" : :descendant, :input, :textarea)[
      !XPath.attr(:type).one_of('submit', 'image', 'radio', 'checkbox', 'hidden', 'file')
    ]
    locate_field(xpath, locator, options)
  end

  expression_filter(:type) do |expr, type|
    type = type.to_s
    if type == 'textarea'
      expr.self(type.to_sym)
    else
      expr[XPath.attr(:type) == type]
    end
  end

  filter_set(:_field, %i[disabled multiple name placeholder])

  node_filter(:with) do |node, with|
    val = node.value
    (with.is_a?(Regexp) ? val =~ with : val == with.to_s).tap do |res|
      add_error("Expected value to be #{with.inspect} but was #{val.inspect}") unless res
    end
  end

  describe_expression_filters
  describe_node_filters do |**options|
    " with value #{options[:with].to_s.inspect}" if options.key?(:with)
  end
end

Capybara.add_selector(:radio_button) do
  label 'radio button'

  xpath do |locator, allow_self: nil, **options|
    xpath = XPath.axis(allow_self ? :"descendant-or-self" : :descendant, :input)[
      XPath.attr(:type) == 'radio'
    ]
    locate_field(xpath, locator, options)
  end

  filter_set(:_field, %i[checked unchecked disabled name])

  node_filter(:option) do |node, value|
    val = node.value
    (val == value.to_s).tap do |res|
      add_error("Expected option value to be #{value.inspect} but it was #{val.inspect}") unless res
    end
  end

  describe_expression_filters
  describe_node_filters do |option: nil, **|
    " with value #{option.inspect}" if option
  end
end

Capybara.add_selector(:checkbox) do
  xpath do |locator, allow_self: nil, **options|
    xpath = XPath.axis(allow_self ? :"descendant-or-self" : :descendant, :input)[
      XPath.attr(:type) == 'checkbox'
    ]
    locate_field(xpath, locator, options)
  end

  filter_set(:_field, %i[checked unchecked disabled name])

  node_filter(:option) do |node, value|
    val = node.value
    (val == value.to_s).tap do |res|
      add_error("Expected option value to be #{value.inspect} but it was #{val.inspect}") unless res
    end
  end

  describe_expression_filters
  describe_node_filters do |option: nil, **|
    " with value #{option.inspect}" if option
  end
end

Capybara.add_selector(:select) do
  label 'select box'

  xpath do |locator, **options|
    xpath = XPath.descendant(:select)
    locate_field(xpath, locator, options)
  end

  filter_set(:_field, %i[disabled multiple name placeholder])

  node_filter(:options) do |node, options|
    actual = if node.visible?
      node.all(:xpath, './/option', wait: false).map(&:text)
    else
      node.all(:xpath, './/option', visible: false, wait: false).map { |option| option.text(:all) }
    end
    (options.sort == actual.sort).tap do |res|
      add_error("Expected options #{options.inspect} found #{actual.inspect}") unless res
    end
  end

  expression_filter(:with_options) do |expr, options|
    options.inject(expr) do |xpath, option|
      xpath[self.class.all[:option].call(option)]
    end
  end

  node_filter(:selected) do |node, selected|
    actual = node.all(:xpath, './/option', visible: false, wait: false).select(&:selected?).map { |option| option.text(:all) }
    (Array(selected).sort == actual.sort).tap do |res|
      add_error("Expected #{selected.inspect} to be selected found #{actual.inspect}") unless res
    end
  end

  node_filter(:with_selected) do |node, selected|
    actual = node.all(:xpath, './/option', visible: false, wait: false).select(&:selected?).map { |option| option.text(:all) }
    (Array(selected) - actual).empty?.tap do |res|
      add_error("Expected at least #{selected.inspect} to be selected found #{actual.inspect}") unless res
    end
  end

  describe_expression_filters do |with_options: nil, **opts|
    desc = +''
    desc << " with at least options #{with_options.inspect}" if with_options
    desc << describe_all_expression_filters(opts)
    desc
  end

  describe_node_filters do |options: nil, selected: nil, with_selected: nil, **|
    desc = +''
    desc << " with options #{options.inspect}" if options
    desc << " with #{selected.inspect} selected" if selected
    desc << " with at least #{with_selected.inspect} selected" if with_selected
    desc
  end
end

Capybara.add_selector(:datalist_input) do
  label 'input box with datalist completion'

  xpath do |locator, **options|
    xpath = XPath.descendant(:input)[XPath.attr(:list)]
    locate_field(xpath, locator, options)
  end

  filter_set(:_field, %i[disabled name placeholder])

  node_filter(:options) do |node, options|
    actual = node.find("//datalist[@id=#{node[:list]}]", visible: :all).all(:datalist_option, wait: false).map(&:value)
    (options.sort == actual.sort).tap do |res|
      add_error("Expected #{options.inspect} options found #{actual.inspect}") unless res
    end
  end

  expression_filter(:with_options) do |expr, options|
    options.inject(expr) do |xpath, option|
      xpath[XPath.attr(:list) == XPath.anywhere(:datalist)[self.class.all[:datalist_option].call(option)].attr(:id)]
    end
  end

  describe_expression_filters do |with_options: nil, **opts|
    desc = +''
    desc << " with at least options #{with_options.inspect}" if with_options
    desc << describe_all_expression_filters(opts)
    desc
  end

  describe_node_filters do |options: nil, **|
    " with options #{options.inspect}" if options
  end
end

Capybara.add_selector(:option) do
  xpath do |locator|
    xpath = XPath.descendant(:option)
    xpath = xpath[XPath.string.n.is(locator.to_s)] unless locator.nil?
    xpath
  end

  node_filter(:disabled, :boolean) { |node, value| !(value ^ node.disabled?) }
  node_filter(:selected, :boolean) { |node, value| !(value ^ node.selected?) }

  describe_node_filters do |**options|
    desc = +''
    desc << " that is#{' not' unless options[:disabled]} disabled" if options.key?(:disabled)
    desc << " that is#{' not' unless options[:selected]} selected" if options.key?(:selected)
    desc
  end
end

Capybara.add_selector(:datalist_option) do
  label 'datalist option'
  visible(:all)

  xpath do |locator|
    xpath = XPath.descendant(:option)
    xpath = xpath[XPath.string.n.is(locator.to_s) | (XPath.attr(:value) == locator.to_s)] unless locator.nil?
    xpath
  end

  node_filter(:disabled, :boolean) { |node, value| !(value ^ node.disabled?) }

  describe_node_filters do |**options|
    " that is#{' not' unless options[:disabled]} disabled" if options.key?(:disabled)
  end
end

Capybara.add_selector(:file_field) do
  label 'file field'
  xpath do |locator, allow_self: nil, **options|
    xpath = XPath.axis(allow_self ? :"descendant-or-self" : :descendant, :input)[
      XPath.attr(:type) == 'file'
    ]
    locate_field(xpath, locator, options)
  end

  filter_set(:_field, %i[disabled multiple name])

  describe_expression_filters
end

Capybara.add_selector(:label) do
  label 'label'
  xpath(:for) do |locator, options|
    xpath = XPath.descendant(:label)
    unless locator.nil?
      locator_matchers = XPath.string.n.is(locator.to_s) | (XPath.attr(:id) == locator.to_s)
      locator_matchers |= XPath.attr(test_id) == locator if test_id
      xpath = xpath[locator_matchers]
    end
    if options.key?(:for) && !options[:for].is_a?(Capybara::Node::Element)
      with_attr = XPath.attr(:for) == options[:for].to_s
      labelable_elements = %i[button input keygen meter output progress select textarea]
      wrapped = !XPath.attr(:for) &
                XPath.descendant(*labelable_elements)[XPath.attr(:id) == options[:for].to_s]
      xpath = xpath[with_attr | wrapped]
    end
    xpath
  end

  node_filter(:for) do |node, field_or_value|
    # Non element values were handled through the expression filter
    next true unless field_or_value.is_a? Capybara::Node::Element

    if (for_val = node[:for])
      field_or_value[:id] == for_val
    else
      field_or_value.find_xpath('./ancestor::label[1]').include? node.base
    end
  end

  describe_expression_filters do |**options|
    " for element with id of \"#{options[:for]}\"" if options.key?(:for) && !options[:for].is_a?(Capybara::Node::Element)
  end
  describe_node_filters do |**options|
    " for element #{options[:for]}" if options[:for]&.is_a?(Capybara::Node::Element)
  end
end

Capybara.add_selector(:table) do
  xpath do |locator, caption: nil, **|
    xpath = XPath.descendant(:table)
    unless locator.nil?
      locator_matchers = (XPath.attr(:id) == locator.to_s) | XPath.descendant(:caption).is(locator.to_s)
      locator_matchers |= XPath.attr(test_id) == locator if test_id
      xpath = xpath[locator_matchers]
    end
    xpath = xpath[XPath.descendant(:caption) == caption] if caption
    xpath
  end

  describe_expression_filters do |caption: nil, **|
    " with caption \"#{caption}\"" if caption
  end
end

Capybara.add_selector(:frame) do
  xpath do |locator, name: nil, **|
    xpath = XPath.descendant(:iframe).union(XPath.descendant(:frame))
    unless locator.nil?
      locator_matchers = (XPath.attr(:id) == locator.to_s) | (XPath.attr(:name) == locator.to_s)
      locator_matchers |= XPath.attr(test_id) == locator if test_id
      xpath = xpath[locator_matchers]
    end
    xpath[find_by_attr(:name, name)]
  end

  describe_expression_filters do |name: nil, **|
    " with name #{name}" if name
  end
end

Capybara.add_selector(:element) do
  xpath do |locator, **|
    XPath.descendant.where(locator ? XPath.local_name == locator.to_s : nil)
  end

  expression_filter(:attributes, matcher: /.+/) do |xpath, name, val|
    builder(xpath).add_attribute_conditions(name => val)
  end

  node_filter(:attributes, matcher: /.+/) do |node, name, val|
    next true unless val.is_a?(Regexp)

    (node[name] =~ val).tap do |res|
      add_error("Expected #{name} to match #{val.inspect} but it was #{node[name]}") unless res
    end
  end

  describe_expression_filters do |**options|
    booleans, values = options.partition { |_k, v| [true, false].include? v }.map(&:to_h)
    desc = describe_all_expression_filters(values)
    desc + booleans.map do |k, v|
      v ? " with #{k} attribute" : "without #{k} attribute"
    end.join
  end
end
# rubocop:enable Metrics/BlockLength
